/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id:  $:

  Description:
    Definition file of the netX Connector API

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2009-11-26  review
    2009-06-04  initial version

**************************************************************************************/

/*****************************************************************************/
/*! \file ConnectorAPI.h
*   Definition file of the netX Connector API
/*****************************************************************************/

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_API netX Connector API implementation         */
/*! \{                                                                       */
/*****************************************************************************/

/* prevent multiple inclusion */
#ifndef __netX_CONNECTOR_API_H
#define __netX_CONNECTOR_API_H

#ifdef __cplusplus
  extern "C" {
#endif  /* _cplusplus */

#ifndef APIENTRY
  #define APIENTRY
#endif

/*****************************************************************************/
/*! Global definitions                                                       */
/*****************************************************************************/
#define MODULE_NAME_EXT                 _T(".dll")    /*!< File extension of connectors                        */
#define NXCON_DLL_SEARCH_STRING        _T("*.dll")    /*!< Search string for connector file names              */
#define NXCON_DLL_PLUGIN_DIR         _T("Plugins")    /*!< Directory containing the connector libraries        */

#define NXCON_FILE_NAME_LENGTH                 256    /*!< Maximum length of a connectors file name            */
#define NXCON_MAX_LENGTH_CONNECTOR_IDENTIFIER    6    /*!< Maximum length of the connector identifier          */
#define NXCON_DESCRIPTION_LENGTH                64    /*!< Maximum length of a connector description           */

#define NXCON_DEFAULT_INTERFACE_NAME_LEN        32    /*!< Maximum length of a interface name                  */
#define NXCON_DEFAULT_HUMANREADABLE_NAME_LEN    64    /*!< Maximum length of the interface human readable name */

#define NXCON_UUID_STRING_SIZE                  37    /*!< "12345678-1234-1234-1234-123456789012 + \0"         */

#define NXCON_MAX_INTFACE_CNT                   99    /*!< Maximum allowed count of interfaces per connector   */

/*****************************************************************************/
/*! Definitions                                                              */
/*****************************************************************************/

/*! Signals for interface notification */
typedef enum NETX_INTERFACE_NOTIFICATION_Etag
{
  eATTACHED,                          /*!< New Interface attached (e.g. plugged usb)   */
  eDETACHED,                          /*!< Interface detached (e.g. unplugged usb)     */
  eDELETED,                           /*!< Interface deleted (e.g. Connector reconfig) */
} NETX_INTERFACE_NOTIFICATION_E;

typedef struct INTERFACE_NAME_TRANSLATION_Ttag
{
  const char*   szSource;
  char*         szDestination;
  unsigned long ulDestinationSize;

} INTERFACE_NAME_TRANSLATION_T;

/*! Commands used with netXConGetInformation() to query connector information */
typedef enum NETX_CONNECTOR_INFO_Etag
{
  eSTATE,                             /*!< Get state of connector ( Enable / Disable )    */
  eUUID,                              /*!< Get UUID of the connector                      */
  eTYPE,                              /*!< Get type of the connector                      */
  eESTABLISH_TIMEOUT,                 /*!< Get timeout for intial device query            */
  eIDENTIFIER,                        /*!< Get connector identifier (e.g. COM, TCP)       */

  eSHORT_INTERFACE_NAME,              /*!< Interface name (e.g. TCP0)                     */
  eLONG_INTERFACE_NAME,               /*!< Human readable name (e.g. IP192.168.1.2:50111) */
  eDESCRIPTION,                       /*!< Get description of the connector               */

} NETX_CONNECTOR_INFO_E;

/*! Commands used with netXConGetInformationInterface() to query interface information */
typedef enum NETX_INTERFACE_INFO_Etag
{
  eINTERFACE_STATE,                   /*!< Get state of interface                         */
  eSEND_TIMEOUT,                      /*!< Get send timeout                               */
  eRESET_TIMEOUT,                     /*!< Get reset timeout                              */
  eKEEPALIVE_TIMEOUT,                 /*!< Get keep alive timeout                         */

} NETX_INTERFACE_INFO_E;

/*! Commands used with netXConGetConfig() / netXConSetConfig() to query/set connector configuration */
typedef enum NETX_CONNECTOR_CONFIG_CMD_Etag
{
  eCMD_CONFIG_GETLEN,                 /*!< Get length of global configuration string      */
  eCMD_CONFIG_GETSTRING,              /*!< Get global configuration                       */
  eCMD_CONFIG_SETSTRING,              /*!< Set global configuration                       */
  eCMD_DIALOG_GETLEN,                 /*!< Get length of configuration string from dialog */
  eCMD_DIALOG_GETSTRING,              /*!< Get configuration string from dialog           */
  eCMD_DIALOG_SETSTRING,              /*!< Set configuration of dialog                    */

} NETX_CONNECTOR_CONFIG_CMD_E;

/*! List of interface communication states */
typedef enum NETX_INTERFACE_STATE_Etag
{
  eINTERFACE_STATE_NOT_SET,           /*!< Interface invalid, no information are valid   */
  eINTERFACE_STATE_NOT_INIT,          /*!< Interface object created, but not initialized */
  eINTERFACE_STATE_INITIALIZED,       /*!< Interface is initialized                      */
  eINTERFACE_STATE_AVAILABLE,         /*!< Interface is available, but not running       */
  eINTERFACE_STATE_NOT_AVAIL,         /*!< Interface not mor available                   */
  eINTERFACE_STATE_RUNNING,           /*!< Interface is ready to handle communication    */
  eINTERFACE_STATE_STOPPED,           /*!< Interface was stopped                         */
  eINTERFACE_STATE_DISABLED,          /*!< Interface is disabled by user                 */

} NETX_INTERFACE_STATE_E;

/*! Union holding the state of a connector */
typedef union NETX_CONNECTOR_INFO_STATE_Utag
{
  struct NETX_CONNECTOR_INFO_STATE_BF_Ttag
  {
    unsigned int  fEnable    : 1;     /*!< Flag, to signal state of connector */
    unsigned int  ulReserved : 31;    /*!< Reserved for future                */
  } tBfState;
  unsigned long   ulState;            /*!< State of connector                 */

} NETX_CONNECTOR_INFO_STATE_U;


#define PCONNECTOR_INTERFACE  void*   /*!< Common datatype for connector interfaces */

/*****************************************************************************/
/*  Connecter API Functions                                                  */
/*****************************************************************************/
/*! Connector callback to notify upper layers about interface attachment/detachment */
typedef void (*PFN_NETXCON_DEVICE_NOTIFY_CALLBACK)     ( const char* szDeviceName, NETX_INTERFACE_NOTIFICATION_E eNotify, void* pvLayer, void* pvUser);

/*! Connector callback to pass received data to upper layers */
typedef void (*PFN_NETXCON_DEVICE_RECEIVE_CALLBACK)    ( unsigned char* pabRxBuffer, unsigned long ulReadLen, void* pvUser);

/* Connector interface functions */
/*! \cond NoDoc */
typedef long                  (APIENTRY *PFN_NETXCON_GETIDENTIFIER)       ( char* szIdentifier, void* pvUUID);
typedef long                  (APIENTRY *PFN_NETXCON_OPEN)                ( PFN_NETXCON_DEVICE_NOTIFY_CALLBACK pfnDevNotifyCallback, void* pvUser);
typedef long                  (APIENTRY *PFN_NETXCON_CLOSE)               ( void);
typedef PCONNECTOR_INTERFACE  (APIENTRY *PFN_NETXCON_CREATEINTERFACE)     ( const char* szDeviceName);
typedef long                  (APIENTRY *PFN_NETXCON_GETINFORMATION)      ( NETX_CONNECTOR_INFO_E eCmd, unsigned long ulSize, void* pvConnectorInfo);
typedef long                  (APIENTRY *PFN_NETXCON_GETCONFIG)           ( NETX_CONNECTOR_CONFIG_CMD_E eCmd, void* pvConfig);
typedef long                  (APIENTRY *PFN_NETXCON_SETCONFIG)           ( NETX_CONNECTOR_CONFIG_CMD_E eCmd, const char* szConfig);

typedef long                  (APIENTRY *PFN_NETXCON_INTF_START)          ( PCONNECTOR_INTERFACE pvInterface, PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveData, void* pvUser);
typedef long                  (APIENTRY *PFN_NETXCON_INTF_STOP)           ( PCONNECTOR_INTERFACE pvInterface);
typedef long                  (APIENTRY *PFN_NETXCON_INTF_SEND)           ( PCONNECTOR_INTERFACE pvInterface, unsigned char* pabData, unsigned long ulDataLen);
typedef long                  (APIENTRY *PFN_NETXCON_INTF_GETINFORMATION) ( PCONNECTOR_INTERFACE pvInterface, NETX_INTERFACE_INFO_E eCmd, unsigned long ulSize, void* pvInterfaceInfo);

typedef HWND                  (APIENTRY *PFN_NETXCON_CREATEDIALOG)        ( HWND hParentWnd, const char* szConfig);
typedef void                  (APIENTRY *PFN_NETXCON_ENDDIALOG)           ( void);
/*! \endcond */

/*! Connector interface function table */
typedef struct NETX_CONNECTOR_FUNCTION_TABLEtag
{
  PFN_NETXCON_GETIDENTIFIER       pfnNetXConGetIdentifier;          /*!< Get identifier of the connector (e.g. COM) */
  PFN_NETXCON_OPEN                pfnNetXConOpen;                   /*!< Open connector                             */
  PFN_NETXCON_CLOSE               pfnNetXConClose;                  /*!< Close connector                            */
  PFN_NETXCON_CREATEINTERFACE     pfnNetXConCreateInterface;        /*!< Create a device interface                  */
  PFN_NETXCON_GETINFORMATION      pfnNetXConGetInformation;         /*!< Request connector information              */
  PFN_NETXCON_GETCONFIG           pfnNetXConGetConfig;              /*!< Get connector configuration                */
  PFN_NETXCON_SETCONFIG           pfnNetXConSetConfig;              /*!< Set connector configuration                */

  PFN_NETXCON_INTF_START          pfnNetXConIntfStart;              /*!< Start connector interface                  */
  PFN_NETXCON_INTF_STOP           pfnNetXConIntfStop;               /*!< Stop connector interface                   */
  PFN_NETXCON_INTF_SEND           pfnNetXConIntfSend;               /*!< Send data via the connector interface      */
  PFN_NETXCON_INTF_GETINFORMATION pfnNetXConIntfGetInformation;     /*!< Request interface information              */

  PFN_NETXCON_CREATEDIALOG        pfnNetXConCreateDialog;           /*!< Create configuration dialog of connector  */
  PFN_NETXCON_ENDDIALOG           pfnNetXConEndDialog;              /*!< Remove configuration dialog of connector  */

} NETX_CONNECTOR_FUNCTION_TABLE, *PNETX_CONNECTOR_FUNCTION_TABLE;

/*! Union holding the type of the connector */
typedef union NETX_CONNECTOR_TYPE_Utag
{
  struct NETX_CONNECTOR_TYPE_BF_Ttag
  {
    unsigned int  fNXDRV_TYPE_DPM       : 1;     /*!< Flag, to signal connector type DPM      */
    unsigned int  fNXDRV_TYPE_USB       : 1;     /*!< Flag, to signal connector type USB      */
    unsigned int  fNXDRV_TYPE_SERIAL    : 1;     /*!< Flag, to signal connector type SERIAL   */
    unsigned int  fNXDRV_TYPE_ETHERNET  : 1;     /*!< Flag, to signal connector type ETHERNET */
    unsigned int  ulReserved            : 28;    /*!< Reserved for future                     */
  } tBfType;
  unsigned long   ulType;                        /*!< Type of connector                       */

} NETX_CONNECTOR_TYPE_U;

/*! Global connector DLL information structure */
typedef struct NETX_CONNECTOR_DLL_INFO_Ttag
{
  HMODULE                         hDll;                                                   /*!< DLL handle                                           */
  unsigned long                   ulError;                                                /*!< DLL load error                                       */
  char                            szFileName[NXCON_FILE_NAME_LENGTH];                     /*!< File name of the DLL                                 */
  char                            szFullFileName[NXCON_FILE_NAME_LENGTH];                 /*!< Full file name of the DLL                            */
  char                            szConIdentifier[NXCON_MAX_LENGTH_CONNECTOR_IDENTIFIER]; /*!< Identifier of connection, realized by this connector */
  char                            szConnectorUUID[NXCON_UUID_STRING_SIZE];                /*!< UUID as string                                       */
  char                            szDescription[NXCON_DESCRIPTION_LENGTH];                /*!< Connector description                                */
  NETX_CONNECTOR_TYPE_U           tType;                                                  /*!< Type of connector                                    */
  NETX_CONNECTOR_FUNCTION_TABLE   tFncTable;                                              /*!< Function table                                       */

} NETX_CONNECTOR_DLL_INFO_T, *PNETX_CONNECTOR_DLL_INFO_T;

/***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif  /* __netX_CONNECTOR_API_H */

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/